/*
 * User-changeable settings for the FNS
 */

#include <stdlib.h>

#include "db.h"
#include "libfma.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_settings.h"
#include "fms_notify.h"

/*
 * local prototypes
 */
static void fms_set_int_setting(char *name, int *setting, char *val);
static void fms_set___charstar_setting(char *name, char **setting, char *val);
static void fms_check_settings_table(db_table_ptr_t tp, int name_col,
  int val_col, char *name, char **val);

/*
 * Alloc settings struct
 */
void
fms_init_settings_vars()
{
  LF_CALLOC(F.settings, struct fms_settings, 1);
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Initialize all settings to values from DB or default
 */
void
fms_init_settings()
{
  db_table_ptr_t tp;
  int name_col;
  int val_col;
  int *colids;
  char *val;

  val = NULL;

  /* open/create the table */
  tp = lf_open_fms_settings_table(F.database, &colids);
  if (tp == NULL) LF_ERROR(("Error opening settings table"));
  
  /* find which column is which */
  name_col = colids[LF_TABLE_FMS_SETTINGS_PARAM];
  val_col = colids[LF_TABLE_FMS_SETTINGS_VALUE];

#define LF_FMS_ONE_SETTING(name, decl, type, dflt, punits, sunits)	\
        LF_DUP_STRING(val, #dflt);					\
	fms_check_settings_table(tp, name_col, val_col, #name, &val);	\
	fms_set_ ## decl ## _setting(#name, &F.settings->name, val);

  LF_FMS_ALL_SETTINGS
#undef LF_FMS_ONE_SETTING

  LF_FREE(colids);
  db_close_table(tp);

  LF_FREE(val);
  return;

 except:
  fms_perror_exit(1);
}

/*
 * This routine checks the settings database for an override value for
 * a particular setting.  If found, the pointer "val" is changed to point to
 * the value from the database
 */
static void
fms_check_settings_table(
  db_table_ptr_t tp,
  int name_col,
  int val_col,
  char *name,
  char **val)
{
  db_datum_t query;
  db_datum_t **qres;
  int nres;
  int rc;

  /* check database for a new value */
  query.dd_type = DBD_STR;
  query.dbd_str = name;
  rc = db_simple_query(tp, name_col, &query, &qres, &nres);
  if (rc != 0) LF_ERROR(("Error querying settings database"));

  /*
   * If we got a result, save the value and free the result
   */
  if (nres > 0) {
    LF_DUP_STRING(*val, qres[0][val_col].dbd_str);
    db_free_query_res(tp, qres, nres);
  }
  return;

 except:
  fms_perror_exit(1);
}


/*
 * Set a string setting
 */
static void
fms_set___charstar_setting(
  char *name,
  char **setting,
  char *val)
{
  fms_notify(FMS_EVENT_DEBUG, "Setting %s = %s", name, val);

  LF_DUP_STRING(*setting, val);
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Set an integer setting
 */
static void
fms_set_int_setting(
  char *name,
  int *setting,
  char *val)
{
  fms_notify(FMS_EVENT_DEBUG, "Setting %s = %s", name, val);

  *setting = atoi(val);
}
